"use strict";

/**
 * Responsible for keeping track of downloaded content.
 *   - Maintains a manifest of what's downloaded
 *   - Actually performs downloads
 *   - Provides the ability to delete those downloads
 */

define(["util", "storage", "models", "apiclient"],
        function(Util, Storage, models, APIClient) {

    var Downloads = {
        /**
         * Initializes download manager
         */
        init: function() {
            var d = $.Deferred();
            this.contentList = new models.ContentList();
            this._readManifest().always(function()  {
                d.resolve();
            });
            return d.promise();
        },
        /**
         * Writes out the manifest file, which keeps track of downloaded data
         */
        _writeManifest: function() {
            var contentListIds = this.contentList.models.map(function(model) {
                return model.getId();
            });
            var jsonManifest = JSON.stringify(contentListIds);
            return Storage.writeText(this.manifestFilename, jsonManifest);
        },
        /**
         * Reads in a manifest file, which keeps track of downloaded data
         */
        _readManifest: function() {
            var d = $.Deferred();
            Storage.readText(this.manifestFilename).done(function(data)  {
                var contentListIds;
                if (data) {
                    contentListIds = JSON.parse(data);
                }
                var contentListModels = models.TopicTree.getContentItemsByIds(contentListIds);
                _(contentListModels).each(function(model)  {
                    this._setDownloaded(model, true);
                }.bind(this));
                this.contentList = new models.ContentList(contentListModels);
                d.resolve();
            }.bind(this)).fail(function()  {
                d.reject();
            });
            return d.promise();
        },
        _setDownloaded: function(model, downloaded) {
            model.setDownloaded(downloaded);
            while (model = model.getParent()) {
                var downloadCount = model.get("downloadCount");
                if (downloaded) {
                    downloadCount++;
                } else {
                    downloadCount--;
                }
                model.set("downloadCount", downloadCount);
            }
        },
        /**
         * Cancels downloading if it's in progress.
         */
        cancelDownloading: function() {
            models.TempAppState.set("isDownloadingTopic", false);
            if (this.currentProgress) {
                this.currentProgress(undefined, undefined, true);
            }
        },
        /**
         * Used to download either a single content item for all content
         * items underneath the specified topic.
         * onProgress callback is only used for topics and is called for
         * each content item.
         */
        download: function(model, onProgress) {
            if (model.isContent()) {
                return this.downloadContent(model, onProgress);
            } else if (model.isTopic()) {
                return this.downloadTopic(model, onProgress);
            }
            return $.Deferred().resolve().pormise();
        },
        /**
         * Downloads all content items recursively one at a time for
         * the current topic
         */
        downloadTopic: function(topic, onProgress) {
            this.currentProgress = onProgress;
            var d = $.Deferred();
            var downloadedCount = 0;
            models.TempAppState.set("isDownloadingTopic", true);
            if (onProgress) {
                onProgress(null, 0);
            }
            var predicate = function(model)  {return !model.isDownloaded();};
            var seq = topic.enumChildrenGenerator(predicate);
            var downloadOneAtATime = function()  {
                try {
                    var contentItem = seq.next().value;
                    // Allow at most one download at a time.
                    this.downloadContent(contentItem).done(function()  {
                        downloadedCount++;
                        if (onProgress) {
                            onProgress(contentItem, downloadedCount);
                        }
                        // Check for cancel
                        if (!models.TempAppState.get("isDownloadingTopic")) {
                            d.resolve(topic, downloadedCount);
                            delete this.currentProgress;
                            return;
                        }
                        setTimeout(downloadOneAtATime, 1000);
                    }.bind(this)).fail(function()  {
                        d.reject();
                    });
                } catch (e) {
                    // done, no more items in the generator
                    models.TempAppState.set("isDownloadingTopic", false);
                    d.resolve(topic, downloadedCount);
                    delete this.currentProgress;
                }
            }.bind(this);
            downloadOneAtATime();
            return d.promise();
        },
        /**
         * Downloads the file at the specified URL and stores it to the
         * specified filename.
         */
        downloadContent: function(contentItem, onProgress) {
            var d = $.Deferred();
            if (onProgress) {
                onProgress(null, 0);
            }

            var filename = contentItem.getId();
            var handleContentLoaded = function(contentData)  {
                var blob = new Blob([contentData], {type: contentItem.getContentMimeType()});
                Storage.writeBlob(filename, blob).done(function()  {
                    this._addDownloadToManifest(contentItem);
                    if (onProgress) {
                        onProgress(contentItem, 1);
                    }
                    d.resolve(contentItem, 1);
                }.bind(this)).fail(function()  {
                    d.reject();
                });
            }.bind(this);
            if (contentItem.isVideo()) {
                var url = contentItem.getDownloadUrl();
                var req = new XMLHttpRequest({mozSystem: true});
                req.open("GET", url, true);
                req.responseType = "arraybuffer";
                req.onload = function()  {
                    handleContentLoaded(req.response);
                };
                req.onerror = function(e)  {
                    d.reject();
                };
                req.send();
            } else if (contentItem.isArticle()) {
                if (contentItem.get("content")) {
                    // Articles have a content property with the html we want to
                    // download already. It's not loaded in by the topic tree but
                    // when the article is actually loaded.
                    handleContentLoaded(contentItem.get("content"));
                } else {
                    // Sometimes articles are downloaded before they are viewed,
                    // so try to download it here.
                    APIClient.getArticle(contentItem.getId()).done(function(result)  {
                        contentItem.set("content", result.translated_html_content);
                        handleContentLoaded(contentItem.get("content"));
                    }).fail(function()  {
                        return d.reject().promise();
                    });
                }
            }
            return d.promise();
        },
        /**
         * Removes a download from the list of downloaded files and
         * removes the file on disk.
         */
        deleteContent: function(contentItem) {
            var d = $.Deferred();
            var filename = contentItem.getId();
            Storage.delete(filename).done(function()  {
                this._removeDownloadFromManifest(contentItem);
                d.resolve();
            }.bind(this)).fail(function()  {
                d.reject();
            });
            return d.promise();
        },
        /**
         * Adds the specified model to the list of downloaded files
         */
        _addDownloadToManifest: function(model) {
            this._setDownloaded(model, true);
            Util.log('adding model to manifest: ');
            Util.log(model);
            this.contentList.push(model);
            this._writeManifest();
        },
        /**
         * Remove the specified model from the list of downloaded files
         */
        _removeDownloadFromManifest: function(model) {
            this._setDownloaded(model, false);
            Util.log('removing model from manifest: ');
            Util.log(model);
            this.contentList.remove(model);
            this._writeManifest();
        },
        manifestFilename: "download-manifest.json"
    }

    return Downloads;

});
